/*
*******************************************************************************
*
* File:         xmutxwin.c
* Description:  
* Author:       Henrik Liebau
* Created:      10 Apr 2000, 13:36:14
* Modified:     Thu Apr 20 15:25:45 2000 by Henrik Liebau, BVS R&D
* Language:     C Source File
* Package:      
* Status:       
*
* (C) Copyright 2000 by Agilent Technologies, All Rights Reserved.
*
*******************************************************************************
*/

#include <windows.h>
#include <assert.h>
#include <stdio.h>

#include <xtypedef.h>
#include <xmutex.h>

/* ///////////////////////////////////////////////////////////////////////// *
 * local function definitions
 * ///////////////////////////////////////////////////////////////////////// */

/*---------------------------------------------------------------------------*
 * static void PrintErrorMessage (const char * fnName, DWORD errCode)
 *
 * Purpose: Print Error message to stderr
 *---------------------------------------------------------------------------*/
static void PrintErrorMessage (const char * fnName, DWORD errCode)
{
    LPVOID lpMsgBuf;
    FormatMessage( 
      FORMAT_MESSAGE_ALLOCATE_BUFFER | 
      FORMAT_MESSAGE_FROM_SYSTEM | 
      FORMAT_MESSAGE_IGNORE_INSERTS,
      NULL,
      errCode,
      MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), /* Default language*/
      (LPTSTR) &lpMsgBuf,
      0,
      NULL 
      );

    /* Display the string. */
    fprintf (stderr,
       "%s returned (%ld) %s",
       fnName, errCode, (const char *) lpMsgBuf);

    /* Free the buffer */
    LocalFree( lpMsgBuf );
}

/*/////////////////////////////////////////////////////////////////////////////
 * exported function definitions
 *///////////////////////////////////////////////////////////////////////////*/

/******************************************************************************
 * mutex stuff
 *****************************************************************************/

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXCreateMutex (bx_mutextype * newMutex)
 *
 * Purpose: create mutex for exclusive access 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCreateMutex (bx_mutextype * pMutex)
{
  bx_errtype retval = BX_E_OK;
  bx_mutextype theMutex;
  DWORD lastError;
  
  assert (pMutex != NULL);

  /* Win32 call*/
  theMutex = CreateMutex( NULL,   /* no security attributes */
                          FALSE,  /* do not lock initially */
                          NULL ); /* no explicit name */
  
  lastError = GetLastError();
  switch (lastError)
  {
   case ERROR_SUCCESS:
    /* everything was OK!*/
    break;
   case ERROR_ALREADY_EXISTS:
    /* no break intentionally!*/
   case ERROR_INVALID_HANDLE: /* could not create mutex*/
    theMutex = NULL;
    retval = BX_E_TBD_MSG;  /* TODO: find good error message */
    PrintErrorMessage("BestXCreateMutex", lastError);
    break;
  }
  
  (*pMutex) = theMutex;

  return retval;
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDeleteMutex (bx_mutextype * pMutex)
 *
 * Purpose: delete mutex handle;
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDeleteMutex (bx_mutextype * pMutex)
{
  bx_errtype retval = BX_E_OK;
  
  assert (pMutex != NULL);

  if (CloseHandle(*pMutex) == FALSE) /* WIN32 call */
  {
    PrintErrorMessage ("BestXDeleteMutex", GetLastError());
    retval = BX_E_TBD_MSG;  /* TODO: good error message */
  }

  *pMutex = (bx_mutextype) NULL;

  return retval;
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXLockMutex (bx_mutextype theMutex)
 *
 * Purpose: lock and block
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXLockMutex (bx_mutextype theMutex)
{
  bx_errtype retval = BX_E_TBD_MSG; /* TODO: find good error message */

  DWORD result = WaitForSingleObject ( theMutex, INFINITE ); /* WIN32 call */
  switch (result)
  {
   case WAIT_OBJECT_0: /* success!*/
    retval = BX_E_OK;
    break;
   case WAIT_ABANDONED: /* do nothing: lock failed */
   case WAIT_TIMEOUT:
    break;
   case WAIT_FAILED:
    PrintErrorMessage ("BestXLockMutex", GetLastError());
    break;
  }

  return retval;
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXUnlockMutex (bx_mutextype theMutex)
 *
 * Purpose: unlock mutex object, returns BX_E_OK on success
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXUnlockMutex (bx_mutextype theMutex)
{
  BOOL retVal = ReleaseMutex(theMutex); /* WIN32 call */

  if (retVal == 0)
  {
    PrintErrorMessage ("BestXUnlockMutex", GetLastError());
  }

  return (retVal == TRUE) ? BX_E_OK : BX_E_TBD_MSG; /*TODO: find good err msg*/
}

/*****************************************************************************
 * critical sections
 *****************************************************************************/

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXCreateCriticalSection (bx_critsecttype * theCritSect)
 *
 * Purpose: 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXCreateCriticalSection (bx_critsecttype * theCritSect)
{
  InitializeCriticalSection (theCritSect); /* WIN32 call */

  return BX_E_OK;
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXDeleteCriticalSection (bx_critsecttype * theCritSect)
 *
 * Purpose: 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDeleteCriticalSection (bx_critsecttype * theCritSect)
{
  DeleteCriticalSection (theCritSect); /* WIN32 call */

  return BX_E_OK;
}


/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXEnterCriticalSection (bx_critsecttype * theCritSect)
 *
 * Purpose: 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXEnterCriticalSection (bx_critsecttype * theCritSect)
{
  EnterCriticalSection (theCritSect); /* WIN32 call */

  return BX_E_OK;
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXLeaveCriticalSection (bx_critsecttype * theCritSect)
 *
 * Purpose: 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXLeaveCriticalSection (bx_critsecttype * theCritSect)
{
  LeaveCriticalSection (theCritSect); /* WIN32 call */

  return BX_E_OK;
}

